/*
**********************************************************************
*                        License Notice
*
* QSFUtils.java is part of the Quincala Software Project, which is
* aimed at playing, viewing, studying, communicating and publishing
* games; primarily the Quincala games, but in time any game that fits.
*
* (C) Copyright 2010 Ulf Aberg (Åberg) and AB Games Ltd. All Rights
* Reserved.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
* Here is also a copy of the gpl licence version 3 in html format,
* called "gpl-3.0-standalone.html" in the com.quincala.core/docs folder.
*
* This licence does not provide any license or right to use any
* trademark owned by Ulf Aberg (Åberg) or AB Games Ltd in any form or
* media. QUINCALA is a registered trademark owned by Ulf Aberg (Åberg).
*
* The design of the (classic) Quincala board is a Registered Design,
* also owned by Ulf Aberg (Åberg).
*
* Contact Ulf on ulf@quincala.com
*
**********************************************************************
*/


package com.quincala.core;

import java.util.Calendar;
import java.net.*;

public class QSFUtils {


	/*
	 * A collection of static methods to evaluate, validate,
	 * and transform QuStrings
	 */

	/*
	 * this will return true for all valid quStrings
	 * acceptable by QuincalaUI
	 * and keep error message temporarily
	 * make isValidQuString static methods
	 */

	/*
	 * make sure to use i18n compatible code!
	 * http://java.sun.com/docs/books/tutorial/i18n/text/charintro.html
	 */

	/*
	 * QSF versions is
	 */
	public static final int qsfLatestBaseLine = 0;
	public static final int qsfLatestIncrement = 1;

	public static int qsfBaseLineToUse = qsfLatestBaseLine;
	public static int qsfIncrementToUse = qsfLatestIncrement;

	//these are game specific validation data;
	//make some sort of system for other games
	//and mini board too when zoom implemented!
	private static int[] boardDimensions = {13,13};
	//TODO use for determining valid coordinate (input for zoom)
	private static int maxRotation = 3;

	//from Qformulas: Needed, or parse??
	public static final char[] asciiDecimalMap = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
		'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
		'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z'};


	//EXPERIMENT; LOOK OK TO DO SWITCH CASE ON THESE:
	public static final int GAME_STRING = 0;
	public static final int URL_STRING = 1;
	public static final int SPACE_KEY_STRING = 2;
	public static final int BOARD_KEY_STRING = 3;
	public static final int GUI_KEY_STRING = 4;
	public static final int QSF_VERSION_STRING = 5;

	public static final String qsfDelimiter = "&";
	public static final String qsfDelReplacement = "%26&";//needed
	public static final String qsfSplitterExpression = "\\&";
	public static final String[] qsfFrame = {"<", ">"};

	public static final String[] qhfPrefixList =
		{"http://www.quincala.com/q/",//for testing
		"http://www.quincala.com/w/",//the real thing
		"file:/home/ulf/Dropbox/quincala.com/web/q/applettester.html",//off line atom testing
		};

	//identify case insensitive in case of Quincala.com
	//test that /a/? works

	public static final String gdvErrorText = "Sorry, this " +
			"software is too old to read the entered game string " +
			"properly - please update.";

	public static final String qsfvErrorText = "Sorry, this " +
	"software is too old to read all the information in the entered QSF string " +
	"properly - please update.";

	public static final String qsfErrorText = "Sorry, the entered QSF string " +
			"has errors and cannot be loaded.";

	private static final int lineLengthDefault = 50;

	//these define the valid keys for QuStrings:
	private static final String[] definedGuiKeys = {"url"};
	//add "textsize"
	private static final String[] definedBoardKeys = {"zoom","rotation"};
	//change to rot here and in panel!

	private static final String[] definedSpaceKeys = {"title","ply","goto","nav", "loc"};

	public static final String[] definedGameHeads = {"Quincala"};
	//add "Gloife"; use upper case as you want the output
	//add go, chess, Pacru slowly

	public static final String[] validQuincalaVariants =
			{"13", "14", "15",
			//"23", "24", "25",
			"33", "34", "35",
			//"45", "53",
			"KM"//, "K3", "K4", "K5",
			//"MM", "M3", "M4", "M5"
			};
	/*
	 * Differentiate between various GDV!
	 */
	//ADD "P3", "P4", "P5"  for pos edit later!



	/*
	 * In addition to all characters with code points
	 * <33 or >126 the illegal chars below are deleted
	 * and removed from user values
	 *
	 * This is for QSF 1.0 and the list will be
	 * reduced as the array above increases
	 *
	 * Note that chars above should be encoded before tested for
	 * legality, so that list overrides the array below
	 */

	//for html encoding of value strings before output in text panels!
//	private static final char[] safeHtmlChars = {' ', '.', '-'};
	//maybe add to this list after study
	//alpha-numeric is safe too!!
	private static final char[] reservedHtmlChars = {'&', '<', '>', '\"', '\''};
	//include ' to be xhtml/xml compliant, add '@' to avoid email harvesting?
	//think again when Unicode or ASCII > 126

	//% is escape char so don't include in reserved!

	private static final char[] wildChars = {'&', '#'};

	private static final char[] dodgyChars = {'@'};

	/*
	 * Re: the choice of reserved and unsafe characters below:
	 *
	 * In the end I choose to harmonise with URI 2005 and
	 * wikipedia/percent-encoding, the latter claims that
	 *
	 * " ... new schemes ... must, in effect, represent characters from the
	 * unreserved set without translation, and should convert
	 * all other characters to bytes according to UTF-8, and
	 * then percent-encode those values. This requirement was
	 * introduced in January 2005 with the publication of RFC 3986.
	 * URI schemes introduced before this date are not affected."
	 *
	 * I have not been able to confirm this, but since QFS might be
	 * introduced as an URI scheme at some point, I
	 * thought it better to comply already.
	 */

	//these are always encoded in QSF:
	private static final char[] reservedChars = {
		'&', ';', '#',//these have function defined
		 '/', //this is used to identify URL strings
		'(', ')', '[', ']', //URI 2005 and java URI checker allow these
		'!', '$', '+', ',',  ':', '=', '?', '*', '\''
		//the last line chars are reserved in URI 2005
	};
	//the "dodgy" char '@' behaves like a reserved char; include in above?

	private static final char[] unsafeChars = {
		'<', '>', '\"', '|', //these are necessary for email and frames
		'{', '}', '^', '`', //throws URI syntax exception in java, but allowed in browsers
		'\\' //illegal in URI 2005; occurs in local file paths in Windows??!
	};

	/*
	 * use this specifically for cleaning QSF input
	 * Any chars here will be automatically removed
	 * from a QSF string before processing starts
	 */

	//email reply/forwarding sorted :-) also line breaks automatically!



	public static String errorMessage = "";


//	private String[] validTildeHeads = {"~info", "~zoom"};//old type
//	private String[] validHashHeads = {"#cmode"};//never used

	public static String getQSFVersion(){
		/*
		 * for now, just return 0.1
		 * later return the latest, and
		 * enable update/downgrade in copy dialogue?
		 */

		return "=QSF;0.1";
	}


	public static boolean is2009QuString(String quStrCand){
		//identify by the very start of it - no need to split on line breaks!
		if (quStrCand.length() < 5)
			return false;


		boolean result = false;
		String head = quStrCand.substring(0, 3);//looking for "qui"
		char nose = quStrCand.charAt(0); //looking for '~'
		char fourthChar = quStrCand.charAt(3);
		boolean fourthIsDigit = Character.isDigit(fourthChar);

		//temporary  for introducing simple games:
		boolean fourthIsMOrK = (fourthChar == 'M') || (fourthChar == 'K');
		boolean fourthIsOK = fourthIsDigit || fourthIsMOrK;

		//temporarily accept quiMM55 etc: only until can read new qsf strings!
		if ((head.equalsIgnoreCase("qui") && fourthIsOK) || nose == '~')
				result = true;
		//better return true; and return false below



		System.out.println("is2009QuString is " + result);

		return result;
	}

	public static boolean isValidQSFString(String cleanQSFString){
		/*
		 * This uses identification methods, "is ..." strings
		 * then validation methods, "isValid ..." strings
		 *
		 * It assumes that the QSF string is
		 */


//		cleanQSFString = cleanQSFString.trim();
		//is part of cleaning!!


		//empty ones are valid, but will not be acted upon:
		if (cleanQSFString.length() == 0)
			return true;


		//first remove hyperlink prefix:
		if (hasQSFHyperLinkPrefix(cleanQSFString)){
			cleanQSFString = getQHyperLinkQuery(cleanQSFString);
		}

		//then see if it has an initial URL
		if(hasInitialURLString(cleanQSFString)){
			if(!isValidURLString(cleanQSFString)){
				System.out.println("Validation Error URL");
				return false;
			}
		} else {

		//then split and feed single strings through:
			String[] s = cleanQSFString.split(QSFUtils.qsfSplitterExpression);
			int splitCount = s.length;

			for (int i = 0; i < splitCount; i++) {
				//no need to trim since cleaning will have removed all spaces!
				String cand = s[i];

				//empty single strings are valid (as in "&&")
				//but are not acted upon
				if(cand.length() == 0)
					return true;


				int type = getQSFStringType(cand);

				switch (type){
				case URL_STRING:
					if(!isValidURLString(cand)){
						System.out.println("Validation Error URL Single");
						return false;
					}
					break;

				case QSF_VERSION_STRING:
					if(!isValidQSFVersionString(cand)){
						System.out.println("Validation Error QSFV");
						return false;
					}
					break;

				case GAME_STRING:
					if (!isValidGameString(cand)){
						System.out.println("Validation Error Game string");
						return false;
//						type = 0;
					}
					break;
				default://to cover all keyvalue string types
					//only accept 2,3,4:
					if(type < 0 || type > 5){
						System.out.println("Validation Error string type out of range!");
						return false;
					}

					if(!isValidKeyValueString(cand)){
						System.out.println("Validation Error KeyvalueString");
						return false;
					}
					break;
				}//end switch statement
			}//end split
		}//end initial url or not

		return true;
	}

	public static int getQSFStringType(String qString){

		/*
		 * this is the only place to IDENTIFY
		 * QSF string types; used in validation
		 * methods and action methods.
		 */


		int type = -1;

		if(qString.length() == 0)
			return -1;

		if (isURLString(qString)) {
			type = URL_STRING;
		} else if (isQSFVersionString(qString)){
			type = QSF_VERSION_STRING;
		} else  if(isKeyValueString(qString)){
			if (isGUIKey(getKey(qString))){
				type = GUI_KEY_STRING;
			}
			if (isBoardKey(getKey(qString))){
				type = BOARD_KEY_STRING;
			}
			if (isSpaceKey(getKey(qString))){
				type = SPACE_KEY_STRING;
			}

		} else {//for identification!!
			type = GAME_STRING;//INITIALISE AS GAME_STRING?
		}
		return type;
	}

	private static boolean isValidTameURLString(String cand){

		//first check if valid URL string:
		if(!isValidURLString(cand)){
			//error messages in that method!
			return false;
		}
		//then check for wild characters:
		int length = cand.length();
		for (int i = 0; i < length; i++) {
			char c = cand.charAt(i);
			if (isInCharArray(c, wildChars)){
				System.out.println("QSF String Error: " +
				"the string has at least one wild character!");
				return false;
			}
		}
		return true;
	}

	public static boolean isValidURLString(String cand){

		//first double check identification?
		if(!isURLString(cand))
			return false;

		//otherwise make sure cand is at least 4 char long:
//		if (cand.length() < 4)
//			return false;

		if (cand.substring(0, 4).equalsIgnoreCase("www.") && cand.indexOf(".", 4) > -1)
			cand = "http://" + cand;
		//only add http:// if starts with www. and has at least one more dot afterwards.

		try {
			URL testURL = new URL(cand);
			System.out.println("URL passed test as "
					+ testURL.toString());
		} catch (MalformedURLException e) {
			System.out.println("URL Candidate is caught malformed!!");
			return false;
		}

		//sending to browser will throw error if no connection, so not ideal!



		return true;
	}

	public static boolean hasInitialURLString(String cand){

		//use this for composite qsf strings

		int slashInd = cand.indexOf('/');
		//what about local windows paths?
		int ampInd = cand.indexOf('&');
		boolean isURLSlash =  slashInd > -1 && (ampInd == -1 || slashInd < ampInd);
		//i.e. there is a slash and it is in the first single string
		//before the first &

		if (isURLSlash){
			return true;
		} else if (cand.length() < 4)
			return false;

		boolean isURLwww = cand.substring(0, 4).equalsIgnoreCase("www.");
		//also should there be a

		return isURLwww;
	}

	public static boolean isURLString(String urlStringCand){
		//assuming minimum url is www.a.de = 8 characters
//		if (urlStringCand.length() < 8)
//			return false;

		//could be "~/help" too ...

		int slashInd = urlStringCand.indexOf('/');
		//what about local windows paths?
//		int ampInd = urlStringCand.indexOf('&');
		boolean isURLSlash =  slashInd > -1;
		//i.e. there is a slash and it is in the first single string
		//before the first &

		if (isURLSlash){
			return true;
		} else if (urlStringCand.length() < 4)
			return false;

		boolean isURLwww = urlStringCand.substring(0, 4).equalsIgnoreCase("www.")
//		&& urlStringCand.indexOf(".", 4) > -1//is this validation??
		;
		//starts with "www." and has at least one more '.'
		//before the first slash???


//		boolean isURLhttp = urlStringCand.substring(0, 5).equalsIgnoreCase("http:");//http:
//		boolean isURLlocal = urlStringCand.substring(0, 2).equalsIgnoreCase("~/");
////		System.out.println("wwwTest is " + wwwTest);
////		System.out.println("httpTest is " + httpTest);
//		boolean isURLString = isURLwww || isURLhttp || isURLlocal;//allow https??

		return isURLwww;

	}

	public static boolean isKeyValueString(String keyValueCand){
//		int firstEqualSignIndex = keyValueCand.indexOf('=');
//		System.out.println("firstEqualSignIndex is " + firstEqualSignIndex);
		return keyValueCand.indexOf('=') > 0;
		//max characters in a key string is 8 NOPE NOT ANY LONGER
		//12345678= has index 012345678;
		//limiting the search to first index enables the free use of '=' in value strings
		//limiting to 9, allows = in URLs (earliest at index 9: OK?)

	}

	public static boolean isValidKeyValueString(String cand){
		//double check?
		if (!isKeyValueString(cand))
			return false;

		String key = getKey(cand);//depend on qsf version!
		String value = getValue(cand);


		/*
		 * maybe validate on value type;
		 * such as "title type"
		 * integer type, double type?
		 * "coordinate, double type"
		 * "unicode type" etc?
		 */


		//or use isInStringArray??
		if (isGUIKey(key)){
			if (key.equalsIgnoreCase("url")){
				if (!isValidTameURLString(value)){
					System.out.println("QSF String Error: " +
					"url value error");
					return false;
				}
			}
//			type = 2;
		} else if (isBoardKey(key)){
			if (key.equalsIgnoreCase("zoom")){
				if(!isValidCoordDoubleValue(value)){
					System.out.println("QSF String Error: " +
					"zoom value error");
					return false;
				}
			} else if (key.equalsIgnoreCase("rotation")){
				if(!isValidIntValue(value,0,maxRotation)){
					System.out.println("QSF String Error: " +
					"rotation value error");
					return false;
				}
			}

//			type = 3;
		} else if (isSpaceKey(key)){
			if (key.equalsIgnoreCase("title")){
				if (!isValidUSAsciiValue(value)){
					System.out.println("QSF String Error: " +
							"title value error");
					return false;
				}
			}
//			type = 1;
		} else {
			System.out.println("QSF Validation Error: " +
					"Key head is not recognised!");
			return false;
		}




		return true;
	}

	private static boolean isQSFVersionString(String cand){
		//only for identification; not validation!
		return cand.indexOf('=') == 0;
	}

	private static boolean isValidQSFVersionString(String qsfVersionCand){
		//argument is trimmed and verified
		//longer than 2 chars in GUI also here!
//		boolean result = true;

		//double check identification:
		if (!isQSFVersionString(qsfVersionCand))
			return false;

		String[] v = qsfVersionCand.substring(1).split(";");
		if (v.length < 2 || !v[0].equalsIgnoreCase("qsf"))
			return false;

		//also test if in the format int.int !

		//set as current version:
		int baseLine = 0;
		int increment = 1;
		String[] s = v[1].split("\\.");
		int fieldCount = s.length;
		System.out.println("QSF Version fieldCount is " + fieldCount);
		if (fieldCount > 0){
			try {
				baseLine = Integer.parseInt(s[0]);
				System.out.println("baseLine is parsed as " + baseLine);
			} catch (NumberFormatException e) {
				System.out.println("QSF Version Validation Error: " +
						"Baseline is entered as: " + s[0]);
				return false;
			}
		}
		if (fieldCount > 1){
			try {
				increment = Integer.parseInt(s[1]);
				System.out.println("Increment is parsed as " + increment);
			} catch (NumberFormatException e) {
				System.out.println("QSF Version Validation Error: " +
						"Increment is entered as: " + s[1]);
				return false;
			}
		}

		//actually, I should set the current value of baseline and incr
		//here; for validation of the rest of the string!!

		//TODO set qsf version:
		return true;
	}


	public static boolean isValidGameString(String gameStringCand){

		/*
		 * TODO allow (up to?) three fields, after
		 * implementing default environments?
		 * test validity of environment too!
		 * QRules.isValidEnvironment
		 */

		//Actually don't need this one! just return
		//true if reached the end:
//		boolean result = true;//until proven otherwise

		String[] s = gameStringCand.split(";");
		//can have only three fields!
		if(s.length > 4){
			System.out.println("QSF Validation Error: too many fields!");
			return false;
		}


		if (s.length > 0){
			//testing game head
			if (!isInStringArray(s[0], definedGameHeads)){
				System.out.println("QSF Validation Error: Game head not recognised!");
				return false;
			}


		} else {
			//is not valid? or loads default?
		}
		if (s.length > 1){
			if (!isValidEnvironmentString(s[0],s[1])){
				System.out.println("QSF Validation Error: Game environment not recognised!");
				return false;
			}


		}
		if (s.length > 2){
			//test marked ply string validity in terms of digits and letters
			String cand = s[2];
			int plyStrLength = cand.length();
			for (int i = 0; i < plyStrLength; i++) {
				char ch = cand.charAt(i);
				if (ch < '0' || ch > 'z'){
					System.out.println("QSF Validation Error: plystring has illegal characters!");
					return false;
				} else if (ch > '9' && ch < 'A'){
					System.out.println("QSF Validation Error: plystring has illegal characters!");
					return false;
				} else if (ch > 'Z' && ch < 'a'){
					System.out.println("QSF Validation Error: plystring has illegal characters!");
					return false;
				}
			}
			//also check that there is an even number
			//of coordinates (or triple for 3D!)
			StringBuffer b = new StringBuffer(cand);
			for (int i = plyStrLength - 1; i >=0; i--) {
				char ch = b.charAt(i);
				if (ch > 'r')
					b.deleteCharAt(i);
			}
			cand = b.toString();
			int pureLength = cand.length();
			if (pureLength % 2 != 0){
				//dimension for 2?
				System.out.println("QSF Validation Error: " +
						"plystring has illegal number of characters!");
				return false;
			}


		}

		if (s.length > 3){
			/*
			 * test if the game definition string is
			 * in the format int.int
			 */
//			int baseLine = 0;
//			int increment = 0;
			/*
			 * TODO maybe set game def from here?
			 * or should validation be separate from setting it?
			 */
			String[] str = s[3].split("\\.");
			int fieldCount = str.length;
			System.out.println("QSF Version fieldCount is " + fieldCount);
			if (fieldCount > 0){
				try {
					int baseLine = Integer.parseInt(str[0]);
					System.out.println("Game Definition baseLine is parsed as "
							+ baseLine);
				} catch (NumberFormatException e) {
					System.out.println("QSF Validation Error: " +
					"Game Definition string has a non-integer base part!");
					return false;
				}
			}
			if (fieldCount > 1){
				try {
					int increment = Integer.parseInt(str[1]);
					System.out.println("Game Definition increment is parsed as "
							+ increment);
				} catch (NumberFormatException e) {
					System.out.println("QSF Validation Error: " +
					"Game Definition string has a non-integer incremental part!");
					return false;
				}
			}
		}

		return true;
	}

	public static boolean isGUIKey(String keyCandidate){
		for (int i = definedGuiKeys.length - 1; i >= 0; i--) {
			//backwards so need only calc length once
			if(definedGuiKeys[i].equals(keyCandidate))
				return true;
		}
		return false;
	}

	public static boolean isBoardKey(String keyCandidate){
		for (int i = definedBoardKeys.length - 1; i >= 0; i--) {
			//backwards so need only calc length once
			if(definedBoardKeys[i].equals(keyCandidate))
				return true;
		}
		return false;
	}

	public static boolean isSpaceKey(String keyCandidate){
		for (int i = definedSpaceKeys.length - 1; i >= 0; i--) {
			//backwards so need only calc length once
			if(definedSpaceKeys[i].equals(keyCandidate))
				return true;
		}
		return false;
	}

	private static boolean isValidIntValue(String cand, int min, int max){

		int candInt = min - 1;

		//test for integer value
		try {
			candInt = Integer.parseInt(cand);
		} catch (NumberFormatException e) {
			System.out.println("QSF String Error: " +
			"the value should be an integer!");
			return false;
		}
		//test for range
		if (candInt < min || candInt > max){
			//inclusive min max values!
			System.out.println("QSF String Error: " +
			"the value is outside the range " + min + " - "
			+ max);
			return false;
		}
		return true;
	}

	private static boolean isValidCoordDoubleValue(String cand){

		/*
		 * this checks if the value has a
		 * valid coordinate, then a double
		 * delimiter is ';' by default
		 */

		String[] split = cand.split(";");
		if (split.length != 2){
			System.out.println("QSF String Error: " +
					"the value must have two fields!");
			return false;
		}

		//testing for coordinate:
		if (!isCoordStringlet(split[0])){
			//(put this method in QSFUtils?)
			System.out.println("QSF String Error: " +
			"the first field must be a coordinate");
			return false;
		}

		//testing for double
		try {
			double d = Double.parseDouble(split[1]);
			System.out.println("double value passes test as " + d);
		} catch (NumberFormatException e) {
			System.out.println("QSF String Error: " +
			"the second field must be a 'double' value!");
			return false;
		}
		return true;
	}

	private static boolean isValidUSAsciiValue(String cand){

		/*
		 * Hmmr, this should never return false,
		 * as long as the string is cleaned off
		 * out of range characters?
		 *
		 * Should I impose a length limit on these
		 * kinds of values?
		 */

		int length = cand.length();
		for (int i = 0; i < length; i++) {
			char c = cand.charAt(i);
			if(c < '!' || c > '~'){
				System.out.println("QSF String Error: " +
						"value is outside US Ascii range.");
				return false;
			}
		}


		return true;
	}

	public static boolean isInStringArray(String cand, String[] array){

		for (int i = array.length - 1; i >= 0; i--) {
			//backwards so need only calc length once
			if(array[i].equalsIgnoreCase(cand))
				return true;
		}
		return false;
	}

	public static boolean isInIntArray(int cand, int[] array){
		//untested but looks OK
		for (int i = array.length - 1; i >= 0; i--) {
			//backwards so need only calc length once
			if(array[i] == cand)
				return true;
		}
		return false;
	}

	public static boolean isInCharArray(char cand, char[] array){
		//untested but looks OK

		for (int i = array.length - 1; i >= 0; i--) {
			//backwards so need only calc length once
			if(array[i] == cand)
				return true;
		}
		return false;
	}

	public static String getKey(String keyValueQuString){
		//or splitOnPreLimiter(keyValueQuString,"=")[0]
		return keyValueQuString.substring(0, keyValueQuString.indexOf('=')).trim().toLowerCase();
	}

	public static String getValue(String keyValueQuString){
		return keyValueQuString.substring(keyValueQuString.indexOf('=') + 1).trim();
	}

	public static String[] splitOnPreLimiter(String toSplit, String preLimiter){
		/*
		 * "Prelimiter" is used for a string that is used to to split a
		 * given string in two parts only one the first occurrence of the
		 * prelimiter;
		 * the normal split(delimiter, 2?) could be used but
		 * this is a tailor made splitter for QSF:
		 *
		 * it trims both return strings, and
		 * converts the first one to lower case
		 */
		String[] result = new String[2];
		result[0] = toSplit.substring(0, toSplit.indexOf(preLimiter)).trim().toLowerCase();
		result[1] =	toSplit.substring(toSplit.indexOf(preLimiter) + 1).trim();
		for (int i = 0; i < 2; i++) {
			System.out.println("prelimiter split[" + i + "] is " + result[1]);
		}

		return result;
	}

	public static boolean isValidTurnString(String turnString, String maxTurnString){
		boolean result = true;
		//maxTurnString is generated automatically so should be OK
//		System.out.println("maxturnstring is " + maxTurnString.substring(1));
		int maxTurn = -1;
		try {
			maxTurn = Integer.parseInt(maxTurnString.substring(1));
		} catch (NumberFormatException e) {
		System.out.println("QSF Parsing Error: " +
				"isValidTurnString.maxTurnString is not an integer!");
		}
		int turnInt = -1;

		//first check all characters in turnString are digits:
		for (int i = 0; i < turnString.length(); i++) {
			char ch = turnString.charAt(i);
			if(!Character.isDigit(ch)){//this spots all non-digits
				result = false;
			}
		}

		//maybe try catch block will take care of above?
		//if only digits, compare range with maxTurns:
		if (result){
			try {
				turnInt = Integer.parseInt(turnString);
			} catch (NumberFormatException e) {
				System.out.println("QSF Parsing Error: " +
				"isValidTurnString.turnString is not an integer!");
			}
			if (turnInt == -1)
				return false;
			if (turnInt > maxTurn || turnInt < 1)
				result = false;
		}
		return result;
	}

	public static String convertToXMLDateFormat(String stringStartingWithDate){
		/*
		 * Not the most sophisticated converting method, but it
		 * will find rows of 8 digits at the beginning of
		 * strings as in a normal title string and insert
		 * '-' at the position 5 and 7
		 */
		if (stringStartingWithDate.length() == 0)
			return "";
		String candidate = stringStartingWithDate.trim();
		int firstSpaceIndex = candidate.indexOf("_");//since spaces are encoded here!
		if (firstSpaceIndex > 0){//since trimmed above
			StringBuffer possibleDateStringBfr = new StringBuffer(candidate.substring(0, firstSpaceIndex));
			String tailEnd = candidate.substring(firstSpaceIndex);

			if(!isPossibleOldDateString(possibleDateStringBfr.toString()))
				return candidate;
			//only converting dates if they are first and in the format DDDDDDDD (8 digits)

			if (firstSpaceIndex == 8){
				//probably YYYYDDMM
				possibleDateStringBfr.insert(6, '-');
				possibleDateStringBfr.insert(4, '-');

			}
			else if (firstSpaceIndex == 10){
				//probably YYYY.DD.MM
				possibleDateStringBfr.replace(7, 8, "-");
				possibleDateStringBfr.replace(4, 5, "-");
			}

			candidate = possibleDateStringBfr.toString() + tailEnd;
		}



		return candidate;
	}

	private static boolean isPossibleOldDateString(String dateStringCandidate){

		//checking that candidate has all digits
		//note that 2000.10.31 is converted too!

		int length = dateStringCandidate.length();
//		if (length != 8)
//			return false;

		for (int i = 0; i < length; i++) {
			char t = dateStringCandidate.charAt(i);
			if (!Character.isDigit(t) && t != '.')
				return false;
		}



		return true;
	}

//	public static String restoreUserInputStringX(String argument){

		//this is not used any more

		/*
		 * this will prepare a quString for being output to
		 * the stringField for editing
		 * This methods undoes what convertUserInputString
		 * does to the user input
		 */


//		String returnString = "";
//		System.out.println("restoreUserInputString is run!");
//
//		StringBuffer testBuffer = new StringBuffer(argument);
//
//		for (int i = testBuffer.length() - 2; i >= 0; i--) {
//			//start 2 from end get at least two chars to evaluate!
//			if(testBuffer.charAt(i) == '%'){
//				String hexCandidate = testBuffer.substring(i+ 1, i+ 3);
//				System.out.println("restore.hexCandidate is " + hexCandidate);
//				if (isHexCode(hexCandidate)){
//					int hexInt = Integer.parseInt(hexCandidate, 16);
//					System.out.println("%hexInt is " + hexInt);
//					String userString = "" + (char) hexInt;
//					testBuffer.replace(i, i+ 3, userString);
//				}
//			}
//		}

//		while (pIndex > -1){
//			//here replace it with char
//			if(pIndex + 3 > testBuffer.length()){
//				break;//any later than this should not be replaced
//			}
//			String hexCode = testBuffer.substring(pIndex + 1, pIndex + 3);
//
//			if(!isHexCode(hexCode))
//				continue;
//
//			System.out.println("%hexCode is " + hexCode);
//			int hexInt = Integer.parseInt(hexCode, 16);
//			System.out.println("%hexInt is " + hexInt);
//			String userString = "" + (char) hexInt;
//			testBuffer.replace(pIndex, pIndex + 3, userString);
//			//test index replace!!
//
//			//now maybe silly if single % at end of line; or no valid hex code?
//
//			pIndex = testBuffer.indexOf("%", pIndex + 1);//test while
//		}


//		int initialBufferLength = testBuffer.length();
//		for (int i = initialBufferLength - 3; i >= 0; i--) {
//			//only looking for %XX so start at - 3!
//			char testChar = testBuffer.charAt(i);
//			if (testChar == '%'){
//				System.out.println("% found! ");
//				String hexCode = testBuffer.substring(i+1, i + 3);
//				System.out.println("hexCode is " + hexCode);
//
////				testBuffer.replace(i, i + 1, "%26");
//			}
//
//
//		}


//		return testBuffer.toString();
//	}

	private static boolean isHexCode(String cand){
		//only pass strings of the format XY; X,Y = 0-9,A-F
		//tests OK!
		//note that it only accepts upper case!!
		//to accept lower case, just do
//		cand = cand.toUpperCase();

		if (cand.length() != 2)
			return false;

		boolean result = true;
		for (int i = 0; i < 2; i++) {
			char ch = cand.charAt(i);
			if (!Character.isDigit(ch) && (ch  < 'A' || ch > 'F'))
				result = false;
		}
		System.out.println("isHexCodeTest comes out " + result);
		return result;
	}

	public static String cleanUserInputValue(String inputString, boolean isUnicode, boolean acceptUnPrintable){
		/*
		 * this will clean the raw input string from GUI text fields
		 * 1. maybe encoding tabs and line breaks to \t and \n
		 * 2. removing out of range characters (isUnicode?)
		 * 3. encode reserved characters! restore restore!!!
		 * 4. removing double spaces
		 * 5. finally trim the string
		 */


		StringBuffer b = new StringBuffer(inputString);
		int codePoint = 0;

		if (acceptUnPrintable){
			for (int i = b.length() - 1; i >= 0; i--) {
				codePoint = b.codePointAt(i);
				//TODO complete action when implemented

			}
		}

		//2. here remove out of range chars

		//is b of new length automatically?
		for (int i = b.length() - 1; i >= 0; i--) {
			codePoint = b.codePointAt(i);
			//TODO test BMP when implemented
			if (codePoint < 32)//allow single spaces
				b.deleteCharAt(i);
			if (!isUnicode && codePoint > 126)
				b.deleteCharAt(i);
			if (isUnicode && codePoint > 65535)//Basic Multilingual Plane
				b.deleteCharAt(i);
		}

		//3. search for double spaces and remove
		int dsIndex = b.indexOf("  ");
		while (dsIndex > -1){
			System.out.println("DOUBLE SPACE is found");
			b.deleteCharAt(dsIndex);
			dsIndex = b.indexOf("  ");
		}

		//4. encode escape and reserved chars, as well as dodgy and unsafe chars!!
		b = encodeValue(b, isUnicode);
//		b = encodeChars(reservedChars, b);
		//use other method?
		//encoding will not create double spaces? NOPE

		return b.toString().trim();
	}

	public static String convertUserInputString(String inputString){

		/*
		 * the user enters a typed string in the stringField with
		 * a prefix such as 'Title:'. This will be converted to
		 * something the gui.quStringGateway can deal with
		 */

		/*
		 * identify user value by local keys, if not
		 * send to cleanWildString!
		 */

		//BUG shouldn't convert single '&'
		//use better recognition of ';;' !!

		if (inputString == null || inputString.length() == 0)
			return "";

		String returnString = inputString;
		//pass it through unless need to change!?

		int colonIndex = inputString.indexOf(':');
		//TODO identify user value strings by localised keys!!
		if (colonIndex > 0){
			//=there is an ':' and it is not first
			//TODO BUG RISK that user input is QSF if no ':'
			//reserve ':' for user level inputs????


			//TODO use splitOnPreLimiter!!
			String head = inputString.substring(0, colonIndex);
			String argument = inputString.substring(colonIndex + 1).trim();
			//leaving out the ':'
//			System.out.println("head is " + head);
//			System.out.println("argument is " + argument);

			/*
			 * develop this to compare a list of
			 * known heads; also standardise on 'title='
			 * as prefix?
			 *
			 * maybe allow translation to compare with
			 * Title button text in local language
			 *
			 * encoded strings/chars are '&&', '%' and maybe '\'
			 */

			if (head.equalsIgnoreCase("title")){
				argument = cleanUserInputValue(argument, false, false);
				//no tabs/linebreak or unicode in title values!!
				returnString = "title=" + argument;
			} 	else if (head.equalsIgnoreCase("zoom")){
				returnString = "zoom=" + argument;
			}

		} else { //treat as bog standard QSF string
			returnString = cleanWildQSFString(returnString);
		}

		return returnString;
	}

	public static String xhtmlEncodeValue(String valueString, boolean isUnicode){

		/*
		 * TODO need to puny code strings first or after for html output??
		 * or will this work for any Unicode character??
		 * Test later!!
		 * for now not valid for Unicode!!
		 */

		StringBuffer b = new StringBuffer();
		int stringLength = valueString.length();
		for (int i = 0; i < stringLength; i++) {
			char ch = valueString.charAt(i);
//			boolean safe = (Character.isLetterOrDigit(ch) || isInCharArray(ch, safeHtmlChars));
			boolean safe = (!isInCharArray(ch, reservedHtmlChars));

			if (safe){
				b.append(ch);
			} else {
				int charInt = (int) ch;
				b.append("&#" + charInt + ";");
			}
		}
//		System.out.println("html converted string is " + b.toString());

		return b.toString();
	}
	//any use to do this using read from buffer??
//	private static StringBuffer htmlEncodeValue(StringBuffer b, boolean isUnicode){
//		/*
//		 * build a similar encoding structure to below
//		 * encoding to "&#decimal;" instead!!
//		 * crap using # - need to stream to buffer!
//		 */
//
//		//first encode all instances of & and ;
//		char[] htmlEscapeChars = {'&', '#', ';'};//move this out later
//		for (int i = b.length() - 1; i >= 0; i--) {
//			char ch = b.charAt(i);
//			if (isInCharArray(ch, htmlEscapeChars)){//isInIntArray better?
//				int charInt = (int) ch;
//				b.replace(i, i + 1, "&#" + charInt + ";");
//			}
//		}
//		//then all instances of non-alphanumeric or space or dot or -.
//
//		for (int i = b.length() - 1; i >= 0; i--) {
//			char ch = b.charAt(i);
//			boolean safe = (Character.isLetterOrDigit(ch) || ch == ' ' || ch == '.' || ch == '-');
//			safe = safe && !isInCharArray(ch, htmlEscapeChars);
//			if (!safe){
//				int charInt = (int) ch;
//				b.replace(i, i + 1, "&#" + charInt + ";");
//			}
//		}
//		System.out.println("html converted string is " + b.toString());
//
//		return b;
//	}

	public static String encodeValue(String rawString, boolean isUnicode){
		StringBuffer b = new StringBuffer(rawString);
		return encodeValue(b, isUnicode).toString();
	}
	private static StringBuffer encodeValue(StringBuffer b, boolean isUnicode){
		//use buffer private static methods!
		//unsafe, reserved, dodgy order?

		//TODO add puny encoding at suitable position!


		b = encodeEscapeSigns(b);
		b = convertSpaces(b);//does encode '_' first

		b = encodeChars(unsafeChars, b);
		b = encodeChars(dodgyChars, b);
		b = encodeChars(reservedChars, b);
		//leaving only safe chars!!


		return b;
	}

	public static String encodeChars(char[] charList, String rawString){
		StringBuffer bfr = new StringBuffer(rawString);
		return encodeChars(charList, bfr).toString();
	}
	private static StringBuffer encodeChars(char[] charList, StringBuffer rawBuffer){

		for (int i = rawBuffer.length() - 1; i >= 0; i--) {
			char ch = rawBuffer.charAt(i);
			if (isInCharArray(ch, charList)){
				String hexCode = Integer.toHexString((int) ch).toUpperCase();
				System.out.println("hexCode is " + hexCode);
				rawBuffer.replace(i, i + 1, "%" + hexCode);

			}
		}
		return rawBuffer;
	}
	private static StringBuffer encodeChar(char charToEncode, StringBuffer rawBuffer){
		String hexCode = Integer.toHexString((int) charToEncode).toUpperCase();
		for (int i = rawBuffer.length() - 1; i >= 0; i--) {
			if (rawBuffer.charAt(i) == charToEncode){
				rawBuffer.replace(i, i + 1, "%" + hexCode);
				System.out.println(charToEncode + " is encoded at " + i);
			}
		}
		return rawBuffer;
	}




	public static String removeChars(char[] charList, String rawBuffer){
		StringBuffer bfr = new StringBuffer(rawBuffer);
		return removeChars(charList, bfr).toString();
	}
	private static StringBuffer removeChars(char[] charList, StringBuffer rawBuffer){

		for (int i = rawBuffer.length() - 1; i >= 0; i--) {
			char ch = rawBuffer.charAt(i);
			if (isInCharArray(ch, charList)){
				rawBuffer.deleteCharAt(i);
			}
		}
		return rawBuffer;
	}

	public static String decodeChars(String rawString){
		StringBuffer bfr = new StringBuffer(rawString);
		return decodeChars(bfr).toString();
	}
	public static StringBuffer decodeChars(StringBuffer bfr){
		/*
		 * this will decode all characters in the form
		 * %HH except %25 which should be done separately?
		 * or just do it here in the right order?
		 * actually by going backwards,
		 * decoded % will not be found!
		 * %2526 will be decoded to %26 not '&'
		 * test without any more
		 */

		System.out.println("decodeChars is run!");

		for (int i = bfr.length() - 2; i >= 0; i--) {
			//start 2 from end get at least two chars to evaluate!
			if(bfr.charAt(i) == '%'){
				String hexCandidate = bfr.substring(i+ 1, i+ 3);
				System.out.println("decodeChars.hexCandidate is " + hexCandidate);
				if (isHexCode(hexCandidate)){
					int hexInt = 0;
					try {
						hexInt = Integer.parseInt(hexCandidate, 16);
					} catch (NumberFormatException e) {
						// should not be caught; is validated in
						//isHexCode
					}
					System.out.println("%hexInt is " + hexInt);
					if (32 < hexInt && hexInt < 127){
						String userString = "" + (char) hexInt;
						bfr.replace(i, i+ 3, userString);
					}
				}
			}
		}
		return bfr;
	}

	public static String decodeValue(String encodedString, boolean isUnicode){
		StringBuffer b = new StringBuffer(encodedString);
		return decodeValue(b, isUnicode).toString();
	}
	private static StringBuffer decodeValue(StringBuffer b, boolean isUnicode){
		//TODO ADD puny code for unicode values!

		b = restoreSpaces(b);
		b = decodeChars(b);

		return b;
	}




	public static String convert2009String(String oldString){

		/*
		 * TODO should I remove ascii 32 from old quStrings
		 * when converting? Wrapping might add such chars?
		 */

//		previous:
//		String longString = oldString;
		//first remove line breaks: by first splitting on line breaks ...
		String[] cutQuString = oldString.split("\r\n|\r|\n");
		int lineCount = cutQuString.length;
		String longString = "";
		for (int i = 0; i < lineCount; i++) {
			longString += cutQuString[i];
		}

		//then convert individual single strings:
		String[] splitall = longString.split(";;");
		int splitCount = splitall.length;
		for (int i = 0; i < splitCount; i++) {
			String head = splitall[i].substring(0,3);//qui?
			System.out.println("convert-head is " + head);
			if(head.equalsIgnoreCase("qui")){
				String environment = splitall[i].substring(3,5);//35
				System.out.println("convert-environment is " + environment);
				String plystring = splitall[i].substring(5).trim();//5544x35 etc
				splitall[i] = "Quincala" + ";" + environment + ";" + plystring + ";0.1";
				System.out.println("convert-plystring is " + plystring);

			} else if(head.equalsIgnoreCase("~in")){//~info
				String body = splitall[i].substring(5).trim();
				body = convertSpaces(body);
				body = convertToXMLDateFormat(body);

				System.out.println("convert-body is:" + body);
				splitall[i] = "title=" + body;
			}
		}
		String newString = "";
		for (int i = 0; i < splitCount; i++) {
			if (i > 0)
				newString += "&&";
			newString += splitall[i];

		}
		System.out.println("convert-newString is " + newString);

		return newString;
	}

	public static String convertSpaces(String rawString){


		StringBuffer rawStringBuffer = new StringBuffer(rawString);
		return convertSpaces(rawStringBuffer).toString();
	}

	public static StringBuffer convertSpaces(StringBuffer rawStringBuffer){
		/*
		 * this converts all ascii 32 spaces to underscores
		 * be careful to encode all real underscores first
		 * trim and run remove double spaces
		 * first!
		 */
		rawStringBuffer = encodeChar('_', rawStringBuffer);

		for (int i = rawStringBuffer.length() - 1; i >= 0; i--) {
			if(rawStringBuffer.codePointAt(i) == 32)
				rawStringBuffer.setCharAt(i, '_');
		}

		return rawStringBuffer;
	}

	//TODO maybe param type StringBuffer to optimise?

	public static String restoreSpaces(String scoreString){
		StringBuffer scoreStringBuffer = new StringBuffer(scoreString);
		return restoreSpaces(scoreStringBuffer).toString();
	}

	public static StringBuffer restoreSpaces(StringBuffer scoreStringBuffer){
		/*
		 * this restores all underscores to ascii 32
		 * be careful to decode all real underscores after this
		 * done in decodeChars!
		 */


		for (int i = scoreStringBuffer.length() - 1; i >= 0; i--) {
			if(scoreStringBuffer.codePointAt(i) == 95)
				scoreStringBuffer.setCharAt(i, ' ');
		}

		return scoreStringBuffer;
	}


	private static StringBuffer removeIllegalPercentEncodings(StringBuffer encodedStringBuffer){

		//here remove instances of %XY where XY is >126 or <33
		//33 to avoid spaces being smuggled in via %HH!

//		StringBuffer testBuffer = new StringBuffer(encodedStringBuffer);

		//not used: simply don't decode %XY where XY is >126 or <33

		for (int i = encodedStringBuffer.length() - 2; i >= 0; i--) {
			//start 2 from end get at least two chars to evaluate!
			if(encodedStringBuffer.charAt(i) == '%'){
				String hexCandidate = encodedStringBuffer.substring(i + 1, i + 3);
				System.out.println("asciifyEncodedStringBfr.hexCandidate is " + hexCandidate);
				if (isHexCode(hexCandidate)){
					//TODO try catch block??
					int hexInt = Integer.parseInt(hexCandidate, 16);
					System.out.println("%XYhexInt is " + hexInt);
					if (hexInt < 33 || hexInt > 126)
						encodedStringBuffer.delete(i, i + 3);
				}
			}
		}

		return encodedStringBuffer;
	}

	public static String cleanWildQSFString(String wildQSFString){
		/*
		 * this is cleaning a QSF string that has been pasted or input through
		 * hyperlink or text field
		 * It removes characters that might have been accidentally
		 * or wrongfully added to a properly output QSF string
		 * or not correctly typed if made manually
		 *
		 * removing all chars <33 incl spaces! and >126
		 * as well as illegal and encoded characters in natura
		 * ascii cleaning?? too
		 *
		 */
		StringBuffer b = new StringBuffer(wildQSFString);
		/*
		 * remove illegal encodings first, just in case
		 * removing other chars will accidentally create
		 * what looks like illegal encoding, e.g.
		 * <title=100% EFFECTIVE> HAS "%EF" after space is removed!
		 * and yes, next time?? but then it wasn't a well formed
		 * QSF string in the first place!
		 */
		//not used: simply don't decode %XY where XY is >126 or <33
//		b = removeIllegalPercentEncodings(b);

		boolean legal;
		char c;
		for (int i = b.length() - 1; i >= 0; i--) {
			legal = true;
			c = b.charAt(i);
			/*
			 *defining USASCII range without spaces and delete char:
			 *also cleaning off line breaks! test again! in all OS!
			 */
			if (c < '!' || c > '~')
				legal = false;

			/*
			 * once I have decided on all userCharsToEncode
			 * illegalChars is not needed.
			 * I want all USASCII characters to be legal
			 * on user level in the future!!
			 */
			if (isInCharArray(c, unsafeChars))
				legal = false;

//			if (isInCharArray(c, userCharsToEncode))
//				legal = false;

			//etc; finally
			if (!legal)
				b.deleteCharAt(i);
		}

		//also encode dodgy chars
		b = encodeChars(dodgyChars, b);

		return b.toString().trim();
	}

	public static String encodeQSFDelimiter(String valueString){
		if (valueString.indexOf(qsfDelimiter) > -1){//>10 enough??
			//encode all instances of && to %26&, so becomes %26%26&!
			StringBuffer temp = new StringBuffer(valueString);
			int daIndex = temp.indexOf(qsfDelimiter);
			while (daIndex > -1) {
//				System.out.println("&& is found in Initial URL");
				temp.replace(daIndex, daIndex + 2, qsfDelReplacement);
				daIndex = temp.indexOf(qsfDelimiter);

				/*
				 * important to start from the beginning every time
				 * the alternative (wrong!) would be?
				 * daIndex = temp.indexOf("&&",daIndex);
				 */

			}
			valueString = temp.toString();
		}

	System.out.println("convQSFString-valueString/URLcandidate is:");
	System.out.println(valueString);
	return valueString;
	}

	public static String encode2009Delimiter(String valueString){
		if (valueString.indexOf(";;") > -1){//>10 enough??
			//encode all instances of && to %26&, so becomes %26%26&!
			StringBuffer temp = new StringBuffer(valueString);
			int daIndex = temp.indexOf(";;");
			while (daIndex > -1) {
//				System.out.println("&& is found in Initial URL");
				temp.replace(daIndex, daIndex + 2,  "%3B;");
				daIndex = temp.indexOf(";;");

				/*
				 * important to start from the beginning every time
				 * the alternative (wrong!) would be?
				 * daIndex = temp.indexOf(";;",daIndex);
				 */

			}
			valueString = temp.toString();
		}

	System.out.println("conv2009String-valueString/URLcandidate is:");
	System.out.println(valueString);
	return valueString;
	}

	public static String encodeEscapeSigns(String valueString){

		System.out.println("encodePercentSign is called");
		StringBuffer b = new StringBuffer(valueString);
		return encodeEscapeSigns(b).toString();
	}
	private static StringBuffer encodeEscapeSigns(StringBuffer b){
		/*
		 * the escape sign, '%', is always encoded,
		 * whether it is used to escape or not.
		 */

		for (int i = b.length() - 1; i >= 0; i--) {
			//must go backwards in order to replace!
			if (b.charAt(i) == '%')
				b.replace(i, i + 1, "%25");
		}
		return b;
	}


	public static boolean isValidEnvironmentString(String gameHead, String envCand){
		//TODO keep making comprehensive environment tests here depending on gameHead
		//Alternatively, do in currentSpace?
		boolean result = true;
		if(gameHead.equalsIgnoreCase("quincala")){
			//here check first two chars
			if(!QSFUtils.isInStringArray(envCand, validQuincalaVariants))
				result = false;
			//also check position, handicap, and rule variants when implemented!
		}
		//else if == chess, gloife etc
		//for chess accept 0-961 etc


		System.out.println("isValidEnvironmentString returns " + result);
		return result;
	}




//	public static boolean isValidTildeString(String qString){
//		boolean result = true;
//
//		return result;
//	}

//	public static boolean isValidQuString(String candidate){
//		boolean result = false;
//
//		//put all the tests here:
//		String headCandidate3 = candidate.substring(0,3);
//		String headCandidate4 = candidate.substring(0,4);
//		String headCandidate5 = candidate.substring(0,5);
//
//
//		//for now only!!
//		result = true;
//		return result;
//	}

	public static String getQHyperLinkQuery(String qsfString){
		String query = "";
		qsfString = qsfString.trim();
		if (hasQSFHyperLinkPrefix(qsfString)){
			System.out.println("QSF String has hyperlink prefix!");
			int qInd = qsfString.indexOf('?');
			if (qInd > -1 && qInd + 1 < qsfString.length()){
				//otherwise no ? or URL ends with '?'
				query = qsfString.substring(qInd + 1);
			}
		}
		return query;
	}

	public static boolean hasQSFHyperLinkPrefix(String qsfString){

		int length = qhfPrefixList.length;
		for (int i = 0; i < length; i++) {
			if (qsfString.indexOf(qhfPrefixList[i]) == 0){
				System.out.println("string has hyperlink prefix!");
				return true;
			}
		}

		return false;
	}

//	public static String convertURLString(String urlCandidate){
//		/*
//		 * is this needed; not so bad to have "&&" any longer?
//		 */
//
//
//		/*
//		 * this methods looks at
//		 * 1. has URL a query bit (there is an '?');
//		 * then either it is a QLink or split on '&'
//		 * and remove all empty strings
//		 * This ensures no instances of '&&' will appear
//		 * 2. Url has no query:
//		 * possible instances of '&&' as in
//		 * http://en.wikipedia.org/wiki/&&
//		 * replace all '&&' with %26&
//		 * http://en.wikipedia.org/wiki/%26& is equivalent!
//		 */
//
//		urlCandidate = urlCandidate.trim();
//		int queryIndex = urlCandidate.indexOf('?');
//		if (queryIndex > -1){//>8 enough, but go for resilience ...
//			//= has a query <=> '&&' means empty key/value pair! badly formed but valid ...
//			String head = urlCandidate.substring(0, queryIndex + 1);
//			//include the '?' in the head!
//			System.out.println("convertingHead is " + head);
//			String tail = "";
//			if (queryIndex + 1 < urlCandidate.length()){
//				//otherwise URL ends with '?'
//				tail = urlCandidate.substring(queryIndex + 1);
//			}
//
//			String[] splitQuery = tail.split("\\&");
//			int keyCount = splitQuery.length;
//			urlCandidate = head;
//			/*
//			 * the following loop reconstructs the URL again,
//			 * leaving out any empty params;
//			 * this should remove all '&&' sequences from query URLs
//			 */
//			int acceptedCount = 0;
//			for (int i = 0; i < keyCount; i++) {
//				if (splitQuery[i].length() > 0){
//					if (acceptedCount > 0)
//						urlCandidate += "&";
//					urlCandidate += splitQuery[i];
//					acceptedCount++;
//				}
//			}
//
//		} else {
//			urlCandidate = encodeQSFDelimiter(urlCandidate);
//		}
////			if (urlCandidate.indexOf("&&") > -1){//>10 enough??
////				//encode all instances of && to %26&, so becomes %26%26&!
////				StringBuffer temp = new StringBuffer(urlCandidate);
////				int daIndex = temp.indexOf("&&");
////				while (daIndex > -1) {
//////					System.out.println("&& is found in Initial URL");
////					temp.replace(daIndex, daIndex + 2, "%26&");
////					daIndex = temp.indexOf("&&");
////
////					/*
////					 * important to start from the beginning every time
////					 * the alternative (wrong!) would be?
////					 * daIndex = temp.indexOf("&&",daIndex);
////					 */
////
////				}
////				urlCandidate = temp.toString();
////			}
////		}
////		System.out.println("QuSTringUtils.convURLString.urlCandidate is:");
////		System.out.println(urlCandidate);
//		return urlCandidate;
//	}

	public static String wrap(String longString){
		return wrap(longString, lineLengthDefault);
	}

	public static String wrap(String longString, int lineLength){
		/*
		 * to avoid a bug with text editors auto-correcting
		 * QSF strings if typing a space or a line break
		 * after them, the wrapper will not wrap a line
		 * leaving less than 5 characters on the last line
		 * 3 characters looks like absolute minimum, 5 or 6 is chosen
		 */


		/*
		 * works in Windows now, but I think it
		 * is impossible to put end of line chars
		 * that works for all operating systems directly
		 * onto the clipboard - the solution seems to
		 * be to put the string onto a text area and
		 * copy using the surrrounding OS! see:
		 * http://www.wilsonmar.com/1eschars.htm#endlinez
		 * On Unix, an ASCII newline (\n) LF (linefeed)  character ends each line.
		 * On Windows, an ASCII sequence CR LF (\r\n) (return followed by linefeed) ends each line.
    	 * On Macintosh, an ASCII CR (\r) (carriage return) character ends each line.
    	 *
    	 * Now, a theory that the toolkit that puts the text on the
    	 * clipboard can determine what OS is there and do it accordingly
    	 * \r\n seems to work well under Windows, but try \n which seems the
    	 * native Java way
		 */

		//if longer than 55
		//break every 50 for length - 5 ...
		int stringLenght = longString.length();
		StringBuffer longStringBfr = new StringBuffer(longString);
		int lineCount = (stringLenght - 6)/ lineLength ;//or 5? OK
		for (int i = lineCount; i > 0 ; i--){
			longStringBfr.insert(i * lineLength, "\n");
		}
		String wrappedString = longStringBfr.toString();


		System.out.println("wrappedString is:");
		System.out.println(wrappedString);
		return wrappedString;
	}

//	public static String putToLowerCase(String mixedCaseString){
//
//		return mixedCaseString.toLowerCase();
//	}

//	public static String ignoreNonAsciiChars(String dirtyString){
//		//this is superceded by the general cleaning method
//		//cleanWildQSFStrings()
//
//		/*
//		 * this method deletes any characters outside the
//		 * inclusive range of 32-126
//		 */
//
//		StringBuffer tempBfr = new StringBuffer(dirtyString);
//
//		//clean!
//		int size = tempBfr.length();
//		for (int i = size - 1; i >= 0; i--) {
//			int testcharInt = (int) tempBfr.charAt(i);
//			if (testcharInt > 126 || testcharInt < 32){
//				//this removes non-descript chars and > ÿ
//				//make Unicode support by only cleaning heads?
//				System.out.println("testcharInt at " + i + " is " + testcharInt);
//
//				tempBfr.deleteCharAt(i);
//
//			}
//
//		}
//
//		return tempBfr.toString();
//	}

//	public static String standarisePassPlies(String oldGameString){
//		/*
//		 * this will replace all "66" plies with "r9"
//		 * making QStrings compatible with Quincala GUIs that
//		 * don't use the centre for passing
//		 */
//		String result = oldGameString;
////		result = result.trim();
////		StringBuffer cand = new StringBuffer(result);
////		//first remova all spaces to make sure plies are at every
////		//two indexes
////		for (int i = cand.length() - 1; i >=0; i--) {
////			char testChar = cand.charAt(i);
////			if (testChar == 32 || testChar == 160)
////				cand.deleteCharAt(i);
////		}
////		//then look for all "66" plies
////		for (int i = cand.length() - 2; i > 3; i-=2) {
////			System.out.println("i is " + i);
////			String testPly = cand.substring(i, i + 2);
////			System.out.println("testPly is " + testPly);
////			if (testPly.equals("66"))
////				cand.replace(i, i + 2, "r9");
////		}
////
////
////
////		result = cand.toString();
////		System.out.println("PASS-converted string is is " + result);
//
//		return result;
//	}

	public static String getIdentityDateString(){
		/*
		 * TODO I18N WARNING!  Is getIDS() this local?
		 * also year 3000 bug!!!
		 *
		 */

//		Date now = new Date();
//		now.toString();//Sat Nov 07 11:54:36 GMT 2009
//		DateFormat idsNow = DateFormat.getDateInstance(DateFormat.SHORT);
//		String dateString = idsNow.format(now);
//		String[] splitdate = dateString.split("/");
//		String idsString = "20" + splitdate[2]+ splitdate[1]+ splitdate[0];

		String idsString = getYearString() + "-" + getMonthString() + "-" +getDateString();

		return idsString;
	}



	public static String getHourString(Calendar now){
		int hourInt = now.get(Calendar.HOUR_OF_DAY);
		String hourString = "" + hourInt;
		if (hourInt < 10)
			hourString = "0" + hourString;
		return hourString;
	}

	public static String getMinuteString(Calendar now){
		int minuteInt = now.get(Calendar.MINUTE);
		String minuteString = "" + minuteInt;
		if (minuteInt < 10)
			minuteString = "0" + minuteString;
		return minuteString;
	}

	public static String getSecondString(Calendar now){
		int secondInt = now.get(Calendar.SECOND);
		String secondString = "" + secondInt;
		if (secondInt < 10)
			secondString = "0" + secondString;
		return secondString;
	}

	public static String getYearString(Calendar now){

//		Calendar nowCalendar = Calendar.getInstance();
		int yearInt = now.get(Calendar.YEAR);
		String yearString = "" + yearInt;

		return yearString;
	}

	public static String getYearString(){

		Calendar nowCalendar = Calendar.getInstance();
		int yearInt = nowCalendar.get(Calendar.YEAR);
		String yearString = "" + yearInt;

		return yearString;
	}

	public static String getMonthString(){
		/*
		 * MONTH returns from 0!!
		 * You should not be using the value returned
		 * by .get(Calendar.MONTH) directly anyway. To
		 * get a date in a particular format you can use
		 * the Format classes in java.text.
		 */
		Calendar nowCalendar = Calendar.getInstance();
		int monthInt = nowCalendar.get(Calendar.MONTH) + 1;
		String monthString = "" + monthInt;
		if (monthInt < 10)
			monthString = "0" + monthString;


		return monthString;
	}

	public static String getMonthString(Calendar now){
		/*
		 * MONTH returns from 0!!
		 * You should not be using the value returned
		 * by .get(Calendar.MONTH) directly anyway. To
		 * get a date in a particular format you can use
		 * the Format classes in java.text.
		 */
//		Calendar nowCalendar = Calendar.getInstance();
		int monthInt = now.get(Calendar.MONTH) + 1;
		String monthString = "" + monthInt;
		if (monthInt < 10)
			monthString = "0" + monthString;


		return monthString;
	}

	public static String getDateString(){

		Calendar nowCalendar = Calendar.getInstance();
		int dateInt = nowCalendar.get(Calendar.DAY_OF_MONTH);
		String dateString = "" + dateInt;
		if (dateInt < 10)
			dateString = "0" + dateString;


		return dateString;
	}

	public static String getDateString(Calendar now){

//		Calendar nowCalendar = Calendar.getInstance();
		int dateInt = now.get(Calendar.DAY_OF_MONTH);
		String dateString = "" + dateInt;
		if (dateInt < 10)
			dateString = "0" + dateString;


		return dateString;
	}

	public static int getAsciiInt(String asciiString){
		//testing: 35 errors on "zz" & 37 errors on > Character.MAX_RADIX
		int result = -1;//returns -1 on parsing error

		try {
			result = Integer.parseInt(asciiString, 36);
		} catch (NumberFormatException e) {
			System.out.println("QSF Parsing Error: value is not an integer!");
		}
		return result;
	}

	public static boolean setQSFVersion(String validatedQSFDefString){
		/*
		 * returns false if error ...
		 * only make this returnable until a separate message
		 * system is in place
		 */

		System.out.println("QSF definitionString received is " + validatedQSFDefString);
		/*
		 * this sets QGD version to latest if not otherwise specified
		 * it sets nothing if version is too high;
		 * a message to update software should be sent
		 * for now this is accomplished by returning false
		 */
		if (validatedQSFDefString.length() == 0){
			qsfBaseLineToUse = qsfLatestBaseLine;
			qsfIncrementToUse = qsfLatestIncrement;
			return true;
		}


		int baseLine = qsfLatestBaseLine;
		int increment = qsfLatestIncrement;
		String[] s = validatedQSFDefString.split("\\.");
		int fieldCount = s.length;
		System.out.println("QSF Version fieldCount is " + fieldCount);
		if (fieldCount > 0)
			try {
				baseLine = Integer.parseInt(s[0]);
			} catch (NumberFormatException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		if (fieldCount > 1)
			try {
				increment = Integer.parseInt(s[1]);
			} catch (NumberFormatException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

//		System.out.println("baseLine is " + baseLine);
//		System.out.println("increment is " + increment);
		if(baseLine > qsfLatestBaseLine || increment > qsfLatestIncrement)
			return false;

		//else:
		qsfBaseLineToUse = baseLine;
		qsfIncrementToUse = increment;

		return true;
	}


	public static String getCoreURLString(String localPath){
		//e.g. "docs/ABGamesLegalCore.html"
		//test with JEditorPane!
		return QSFUtils.class.getResource(localPath).toString();
	}

	//these are probably not needed; rewrite!
	public static String historyURLString(){
		java.net.URL historyURL = QSFUtils.class.getResource("history/qustringhistory.txt");

		return historyURL.toString();
	}



	public static String getSessionPathString(){
		java.net.URL historyURL = QSFUtils.class.getResource("history/");

		return historyURL.toString();
	}

	//from Qformulas:

	public static int hexCharToInt(char hexChar){

//		if (hexChar > 'r')//'r' corresponds to 28 below:
//			return -1;//or other "outside limits" return?

		String hexString = "" + hexChar;
		int i = -1;

		try {
			i = Integer.parseInt(hexString, 36);
		} catch (NumberFormatException e) {
			System.out.println("QUtils Error: " +
			"The Character is not ascii-decimal!");
		}

		return i;

	}

	public static String locToHexCharString(int loc) {
		//this is specific, but could be made general if 13 is a parameter
		int hexNum1 = 0;
		int hexNum2 = 0;
		if (loc < 169){
			hexNum1 = loc % 13;
			hexNum2 = 12 - (loc / 13);
		} else {
			//TODO make r0 out of 810!!
			//opposite of loc = xCoord * 30 + yCoord;
			hexNum1 = loc / 36;
			hexNum2 = loc % 36;//try this
			//asciidecimal!

		}

		// int hexNum2 = loc / 13;
		String hexChars = "" + asciiDecimalMap[hexNum1] + asciiDecimalMap[hexNum2];
//		System.out.println("hexChars String is " + hexChars);
		return hexChars;
	}

	public static int hexStringletToLoc(String stringlet){
		//this should really test for stringlet validity!!!
//		int loc = -1;//or other "outside limits" return?
		char xChar = stringlet.charAt(0);
		char yChar = stringlet.charAt(1);
		return hexCharsToLoc(xChar, yChar);

	}

	public static int hexCharsToLoc(char hexChar1, char hexChar2) {

		int xCoord = hexCharToInt(hexChar1);
		int yCoord = hexCharToInt(hexChar2);

		if (xCoord == -1 || yCoord == -1){
			return -1;
		}

		int loc = -1;//or other "outside limits" return?
		//coordinate with hexCharToInt()
		if (xCoord < 13){//incuding 'c' but not 'd'
			//			this is only for on-the board locs:
			loc = (12 - yCoord) * 13 + xCoord;
		} else {
			loc = xCoord * 36 + yCoord;//:-) coordinate with locToHexCharString()!
		}

		/*
		 * regarding resign locs
		 * r0 => 810
		 * r1 => 811
		 * r2 => 812
		 * r9 => 819 (pass as out-of board ply! - coordinate with recieveEntry!)
		 */

		return loc;
	}

	public static boolean isCoordStringlet(String slet){

		if (slet.length() != 2)
			return false;
		//allow 0-9 and a-c only
		return isCoordChar(slet.charAt(0)) && isCoordChar(slet.charAt(1));

	}

	public static boolean isCoordChar(char a){
		int boardSize = 13;//or get from somewhere!!
		for (int i = 0; i < boardSize; i++) {
			if (a == asciiDecimalMap[i]){
				return true;
			}
		}
		return false;
	}

}
